/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.cassandra.schema;

import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.regex.Pattern;

import com.google.common.collect.ImmutableSet;

import org.apache.cassandra.db.Digest;

import static org.apache.cassandra.db.Directories.TABLE_DIRECTORY_NAME_SEPARATOR;

public final class SchemaConstants
{
    public static final Pattern PATTERN_WORD_CHARS = Pattern.compile("\\w+");
    public static final Pattern PATTERN_NON_WORD_CHAR = Pattern.compile("\\W");

    public static final String SYSTEM_KEYSPACE_NAME = "system";
    public static final String SCHEMA_KEYSPACE_NAME = "system_schema";

    public static final String TRACE_KEYSPACE_NAME = "system_traces";
    public static final String AUTH_KEYSPACE_NAME = "system_auth";
    public static final String DISTRIBUTED_KEYSPACE_NAME = "system_distributed";

    public static final String VIRTUAL_SCHEMA = "system_virtual_schema";

    public static final String VIRTUAL_VIEWS = "system_views";

    /* system keyspace names (the ones with LocalStrategy replication strategy) */
    public static final Set<String> LOCAL_SYSTEM_KEYSPACE_NAMES =
        ImmutableSet.of(SYSTEM_KEYSPACE_NAME, SCHEMA_KEYSPACE_NAME);

    /* replicate system keyspace names (the ones with a "true" replication strategy) */
    public static final Set<String> REPLICATED_SYSTEM_KEYSPACE_NAMES =
        ImmutableSet.of(TRACE_KEYSPACE_NAME, AUTH_KEYSPACE_NAME, DISTRIBUTED_KEYSPACE_NAME);
    /**
     * longest permissible KS or CF name.  Our main concern is that filename not be more than 255 characters;
     * the filename will contain both the KS and CF names. Since non-schema-name components only take up
     * ~64 characters, we could allow longer names than this, but on Windows, the entire path should be not greater than
     * 255 characters, so a lower limit here helps avoid problems.  See CASSANDRA-4110.
     */
    public static final int NAME_LENGTH = 48;

    /**
     * Longest acceptable file name. Longer names lead to too long file name error.
     */
    public static final int FILENAME_LENGTH = 255;

    /**
     * Length of a table uuid as a hex string.
     */
    public static final int TABLE_UUID_AS_HEX_LENGTH = 32;

    /**
     * Longest acceptable table name, so it can be used in a directory
     * name constructed with a suffix of a table id and a separator.
     */
    public static final int TABLE_NAME_LENGTH = FILENAME_LENGTH - TABLE_UUID_AS_HEX_LENGTH - TABLE_DIRECTORY_NAME_SEPARATOR.length();

    // 59adb24e-f3cd-3e02-97f0-5b395827453f
    public static final UUID emptyVersion;

    public static final List<String> LEGACY_AUTH_TABLES = Arrays.asList("credentials", "users", "permissions");

    /**
     * Validates that a name is not empty and contains only alphanumeric characters or
     * underscore, so it can be used in file or directory names.
     *
     * @param name the name to check
     * @return whether the non-empty name contains only valid characters
     */
    public static boolean isValidCharsName(String name)
    {
        return name != null && !name.isEmpty() && PATTERN_WORD_CHARS.matcher(name).matches();
    }

    static
    {
        emptyVersion = UUID.nameUUIDFromBytes(Digest.forSchema().digest());
    }

    /**
     * @return whether or not the keyspace is a really system one (w/ LocalStrategy, unmodifiable, hardcoded)
     */
    public static boolean isLocalSystemKeyspace(String keyspaceName)
    {
        return LOCAL_SYSTEM_KEYSPACE_NAMES.contains(keyspaceName.toLowerCase());
    }

    /**
     * @return whether or not the keyspace is a replicated system ks (system_auth, system_traces, system_distributed)
     */
    public static boolean isReplicatedSystemKeyspace(String keyspaceName)
    {
        return REPLICATED_SYSTEM_KEYSPACE_NAMES.contains(keyspaceName.toLowerCase());
    }

    /**
     * Checks if the keyspace is a virtual system keyspace.
     * @return {@code true} if the keyspace is a virtual system keyspace, {@code false} otherwise.
     */
    public static boolean isVirtualSystemKeyspace(String keyspaceName)
    {
        return VIRTUAL_SCHEMA.equals(keyspaceName.toLowerCase()) || VIRTUAL_VIEWS.equals(keyspaceName.toLowerCase());
    }

    /**
     * Checks if the keyspace is a system keyspace (local replicated or virtual).
     * @return {@code true} if the keyspace is a system keyspace, {@code false} otherwise.
     */
    public static boolean isSystemKeyspace(String keyspaceName)
    {
        return isLocalSystemKeyspace(keyspaceName)
                || isReplicatedSystemKeyspace(keyspaceName)
                || isVirtualSystemKeyspace(keyspaceName);
    }

    /**
     * @return whether or not the keyspace is a non-virtual, system keyspace
     */
    public static boolean isNonVirtualSystemKeyspace(String keyspaceName)
    {
        final String lowercaseKeyspaceName = keyspaceName.toLowerCase();
        return LOCAL_SYSTEM_KEYSPACE_NAMES.contains(lowercaseKeyspaceName)
               || REPLICATED_SYSTEM_KEYSPACE_NAMES.contains(lowercaseKeyspaceName);
    }
}
