/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.cassandra.sidecar.utils;

import java.util.Objects;

import org.junit.jupiter.api.Test;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.assertj.core.api.Assertions.assertThat;

/**
 * Unit tests for {@link FileUtils}
 */
class FileUtilsTest
{
    @Test
    void testMaybeResolveHomeDirectory()
    {
        assertThat(FileUtils.maybeResolveHomeDirectory(null)).isNull();
        assertThat(FileUtils.maybeResolveHomeDirectory("")).isEqualTo("");
        assertThat(FileUtils.maybeResolveHomeDirectory("~")).isEqualTo(System.getProperty("user.home"));
        assertThat(FileUtils.maybeResolveHomeDirectory("~/")).isEqualTo(System.getProperty("user.home") + "/");
        assertThat(FileUtils.maybeResolveHomeDirectory("~/.ccm")).isEqualTo(System.getProperty("user.home") + "/.ccm");
        assertThat(FileUtils.maybeResolveHomeDirectory("/dev/null")).isEqualTo("/dev/null");
    }

    @Test
    public void testStorageStringPatterns()
    {
        assertEquals(1, FileUtils.storageUnitToBytes("B"));
        assertEquals(1024, FileUtils.storageUnitToBytes("KiB"));
        assertEquals(1048576, FileUtils.storageUnitToBytes("MiB"));
        assertEquals(1073741824, FileUtils.storageUnitToBytes("GiB"));

        assertEquals(1048576L, FileUtils.mbStringToBytes("1"));
        assertEquals(524288000L, FileUtils.mbStringToBytes("500"));

        assertEquals(1L, Objects.requireNonNull(FileUtils.storageStringToBytes("1")).longValue());
        assertEquals(1L, Objects.requireNonNull(FileUtils.storageStringToBytes("1B")).longValue());
        assertEquals(500L, Objects.requireNonNull(FileUtils.storageStringToBytes("500B")).longValue());
        assertEquals(1024, Objects.requireNonNull(FileUtils.storageStringToBytes("1KiB")).longValue());
        assertEquals(1048576, Objects.requireNonNull(FileUtils.storageStringToBytes("1024KiB")).longValue());
        assertEquals(1048576, Objects.requireNonNull(FileUtils.storageStringToBytes("1MiB")).longValue());
        assertEquals(4294967296L, Objects.requireNonNull(FileUtils.storageStringToBytes("4096MiB")).longValue());
        assertEquals(536870912000L, Objects.requireNonNull(FileUtils.storageStringToBytes("500GiB")).longValue());
    }
}
