/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.sidecar.restore;

import com.codahale.metrics.Timer;
import com.google.common.collect.Range;
import com.google.common.collect.RangeMap;
import com.google.common.collect.TreeRangeMap;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import io.vertx.core.Future;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.cassandra.sidecar.cluster.ConsistencyVerifier;
import org.apache.cassandra.sidecar.cluster.ConsistencyVerifiers;
import org.apache.cassandra.sidecar.cluster.locator.InstanceSetByDc;
import org.apache.cassandra.sidecar.common.data.ConsistencyVerificationResult;
import org.apache.cassandra.sidecar.common.data.RestoreJobProgressFetchPolicy;
import org.apache.cassandra.sidecar.common.data.RestoreJobStatus;
import org.apache.cassandra.sidecar.common.response.TokenRangeReplicasResponse;
import org.apache.cassandra.sidecar.common.server.cluster.locator.Token;
import org.apache.cassandra.sidecar.common.server.cluster.locator.TokenRange;
import org.apache.cassandra.sidecar.common.server.data.RestoreRangeStatus;
import org.apache.cassandra.sidecar.common.utils.Preconditions;
import org.apache.cassandra.sidecar.common.utils.StringUtils;
import org.apache.cassandra.sidecar.concurrent.ExecutorPools;
import org.apache.cassandra.sidecar.concurrent.TaskExecutorPool;
import org.apache.cassandra.sidecar.db.RestoreJob;
import org.apache.cassandra.sidecar.db.RestoreRange;
import org.apache.cassandra.sidecar.db.RestoreRangeDatabaseAccessor;
import org.apache.cassandra.sidecar.metrics.SidecarMetrics;
import org.apache.cassandra.sidecar.metrics.StopWatch;
import org.apache.cassandra.sidecar.metrics.server.RestoreMetrics;
import org.apache.cassandra.sidecar.restore.RestoreJobDiscoverer;
import org.apache.cassandra.sidecar.restore.RestoreJobProgress;
import org.apache.cassandra.sidecar.restore.RestoreJobProgressCollector;
import org.apache.cassandra.sidecar.restore.RestoreJobProgressCollectors;
import org.apache.cassandra.sidecar.restore.RingTopologyRefresher;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.annotations.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class RestoreJobConsistencyChecker {
    private static final Logger LOGGER = LoggerFactory.getLogger(RestoreJobConsistencyChecker.class);
    private final RingTopologyRefresher ringTopologyRefresher;
    private final RestoreJobDiscoverer restoreJobDiscoverer;
    private final RestoreRangeDatabaseAccessor rangeDatabaseAccessor;
    private final TaskExecutorPool taskExecutorPool;
    private final RestoreMetrics restoreMetrics;
    private volatile boolean firstTimeSinceImportReady = true;

    @Inject
    public RestoreJobConsistencyChecker(RingTopologyRefresher ringTopologyRefresher, RestoreJobDiscoverer restoreJobDiscoverer, RestoreRangeDatabaseAccessor rangeDatabaseAccessor, ExecutorPools executorPools, SidecarMetrics sidecarMetrics) {
        this.ringTopologyRefresher = ringTopologyRefresher;
        this.restoreJobDiscoverer = restoreJobDiscoverer;
        this.rangeDatabaseAccessor = rangeDatabaseAccessor;
        this.taskExecutorPool = executorPools.internal();
        this.restoreMetrics = sidecarMetrics.server().restore();
    }

    public Future<RestoreJobProgress> check(RestoreJob restoreJob, RestoreJobProgressFetchPolicy fetchPolicy) {
        Preconditions.checkArgument((restoreJob.consistencyLevel != null ? 1 : 0) != 0, (String)"Consistency level of the job must present");
        Preconditions.checkArgument((!restoreJob.consistencyLevel.isLocalDcOnly || StringUtils.isNotEmpty((String)restoreJob.localDatacenter) ? 1 : 0) != 0, (String)"When using local consistency level, localDatacenter must present");
        RestoreJobProgressCollector collector = RestoreJobProgressCollectors.create(restoreJob, fetchPolicy);
        RestoreRangeStatus successCriteria = restoreJob.expectedNextRangeStatus();
        ConsistencyVerifier verifier = ConsistencyVerifiers.forConsistencyLevel(restoreJob.consistencyLevel, restoreJob.localDatacenter);
        LOGGER.info("Checking restore job progress. jobId={} fetchPolicy={} successCriteria={}", new Object[]{restoreJob.jobId, fetchPolicy, successCriteria});
        Future future = this.ringTopologyRefresher.replicaByTokenRangeAsync(restoreJob).compose(topology -> this.findRangesAndConclude(restoreJob, successCriteria, (TokenRangeReplicasResponse)topology, verifier, collector));
        return StopWatch.measureTimeTaken(future, durationNanos -> ((Timer)this.restoreMetrics.consistencyCheckTime.metric).update(durationNanos, TimeUnit.NANOSECONDS));
    }

    private Future<RestoreJobProgress> findRangesAndConclude(RestoreJob restoreJob, RestoreRangeStatus successCriteria, TokenRangeReplicasResponse topology, ConsistencyVerifier verifier, RestoreJobProgressCollector collector) {
        return this.taskExecutorPool.executeBlocking(() -> {
            short bucketId = 0;
            return this.rangeDatabaseAccessor.findAll(restoreJob.jobId, bucketId);
        }).map(ranges -> {
            if (this.shouldForceRestoreJobDiscoverRun(restoreJob, (List<RestoreRange>)ranges)) {
                this.taskExecutorPool.runBlocking(this.restoreJobDiscoverer::tryExecuteDiscovery, false);
                return RestoreJobProgress.pending(restoreJob);
            }
            RestoreJobConsistencyChecker.concludeRanges(ranges, topology, verifier, successCriteria, collector);
            return collector.toRestoreJobProgress();
        });
    }

    private boolean shouldForceRestoreJobDiscoverRun(RestoreJob restoreJob, List<RestoreRange> ranges) {
        long foundSliceCount = RestoreJobConsistencyChecker.sliceCountFromRanges(ranges);
        if (foundSliceCount < restoreJob.sliceCount) {
            LOGGER.warn("Not all restore ranges are found. Mark the progress as pending and force restore job discover run. jobId={} expectedSliceCount={} foundSliceCount={}", new Object[]{restoreJob.jobId, restoreJob.sliceCount, foundSliceCount});
            return true;
        }
        if (restoreJob.status == RestoreJobStatus.IMPORT_READY && this.firstTimeSinceImportReady) {
            this.firstTimeSinceImportReady = false;
            LOGGER.info("First time checking consistency of the restore job after import_ready. Mark the progress as pending and force restore job discover run. jobId={}", (Object)restoreJob.jobId);
            return true;
        }
        return false;
    }

    private static void concludeRanges(List<RestoreRange> ranges, TokenRangeReplicasResponse topology, ConsistencyVerifier verifier, RestoreRangeStatus successCriteria, RestoreJobProgressCollector collector) {
        RangeMap<Token, InstanceSetByDc> replicasPerRange = RestoreJobConsistencyChecker.populateReplicas(topology);
        Map<Range<Token>, Pair<Map<String, RestoreRangeStatus>, RestoreRange>> statusPerRange = RestoreJobConsistencyChecker.populateStatusByReplica(ranges);
        for (Map.Entry<Range<Token>, Pair<Map<String, RestoreRangeStatus>, RestoreRange>> entry : statusPerRange.entrySet()) {
            if (!collector.canCollectMore()) {
                return;
            }
            Range<Token> tokenRange = entry.getKey();
            Map status = (Map)entry.getValue().getLeft();
            RestoreRange relevantRestoreRange = (RestoreRange)entry.getValue().getRight();
            ConsistencyVerificationResult res = RestoreJobConsistencyChecker.concludeOneRange(replicasPerRange, verifier, successCriteria, tokenRange, status);
            collector.collect(relevantRestoreRange, res);
        }
    }

    static RangeMap<Token, InstanceSetByDc> populateReplicas(TokenRangeReplicasResponse topology) {
        TreeRangeMap replicasPerRange = TreeRangeMap.create();
        for (TokenRangeReplicasResponse.ReplicaInfo replicaInfo : topology.writeReplicas()) {
            TokenRange tokenRange = new TokenRange(Token.from((String)replicaInfo.start()), Token.from((String)replicaInfo.end()));
            Map replicasByDc = replicaInfo.replicasByDatacenter();
            HashMap<String, Set<String>> mapping = new HashMap<String, Set<String>>(replicasByDc.size());
            replicasByDc.forEach((dc, instances) -> mapping.put((String)dc, new HashSet(instances)));
            InstanceSetByDc instanceSetByDc = new InstanceSetByDc(mapping);
            replicasPerRange.put(tokenRange.range, (Object)instanceSetByDc);
        }
        return replicasPerRange;
    }

    static Map<Range<Token>, Pair<Map<String, RestoreRangeStatus>, RestoreRange>> populateStatusByReplica(List<RestoreRange> ranges) {
        TreeRangeMap rangeMap = TreeRangeMap.create();
        rangeMap.put(Range.all(), (Object)Pair.of(Collections.emptyMap(), null));
        for (RestoreRange restoreRange : ranges) {
            Range tokenRange = restoreRange.tokenRange().range;
            RangeMap sub = rangeMap.subRangeMap(tokenRange);
            Map map = sub.asMapOfRanges();
            if (map.isEmpty()) {
                rangeMap.put(tokenRange, (Object)Pair.of(restoreRange.statusByReplica(), (Object)restoreRange));
                continue;
            }
            TreeRangeMap updated = TreeRangeMap.create();
            map.forEach((arg_0, arg_1) -> RestoreJobConsistencyChecker.lambda$populateStatusByReplica$5(restoreRange, (RangeMap)updated, arg_0, arg_1));
            rangeMap.putAll((RangeMap)updated);
        }
        Map result = rangeMap.asMapOfRanges();
        result.entrySet().removeIf(e -> ((Pair)e.getValue()).getValue() == null);
        return result;
    }

    private static ConsistencyVerificationResult concludeOneRange(RangeMap<Token, InstanceSetByDc> replicasByRange, ConsistencyVerifier verifier, RestoreRangeStatus successCriteria, Range<Token> range, Map<String, RestoreRangeStatus> statusByReplica) {
        Map<RestoreRangeStatus, Set<String>> groupByStatus = RestoreJobConsistencyChecker.groupReplicaByStatus(statusByReplica);
        Set<String> succeeded = groupByStatus.getOrDefault(successCriteria, Collections.emptySet());
        Set<String> failed = groupByStatus.getOrDefault(RestoreRangeStatus.FAILED, Collections.emptySet());
        InstanceSetByDc replicaSet = RestoreJobConsistencyChecker.replicaSetForRange(range, replicasByRange);
        if (replicaSet == null) {
            return ConsistencyVerificationResult.PENDING;
        }
        ConsistencyVerificationResult result = verifier.verify(succeeded, failed, replicaSet);
        switch (result) {
            case FAILED: {
                return ConsistencyVerificationResult.FAILED;
            }
            case PENDING: {
                return ConsistencyVerificationResult.PENDING;
            }
        }
        return ConsistencyVerificationResult.SATISFIED;
    }

    private static Map<RestoreRangeStatus, Set<String>> groupReplicaByStatus(Map<String, RestoreRangeStatus> statusMap) {
        return statusMap.entrySet().stream().collect(Collectors.groupingBy(Map.Entry::getValue, Collectors.mapping(Map.Entry::getKey, Collectors.toSet())));
    }

    @Nullable
    private static InstanceSetByDc replicaSetForRange(Range<Token> tokenRange, RangeMap<Token, InstanceSetByDc> replicasByRange) {
        Map subRange = replicasByRange.subRangeMap(tokenRange).asMapOfRanges();
        if (subRange.size() == 1) {
            Map.Entry foundEntry = subRange.entrySet().iterator().next();
            if (((Range)foundEntry.getKey()).encloses(tokenRange)) {
                return (InstanceSetByDc)foundEntry.getValue();
            }
            LOGGER.info("Topology is changed");
            return null;
        }
        LOGGER.warn("Unable to find a single match for range. tokenRange={} matchingRanges={}", tokenRange, (Object)subRange);
        return null;
    }

    private static long sliceCountFromRanges(List<RestoreRange> ranges) {
        return ranges.stream().map(RestoreRange::sliceId).distinct().count();
    }

    @VisibleForTesting
    static ConsistencyVerificationResult concludeOneRangeUnsafe(TokenRangeReplicasResponse topology, ConsistencyVerifier verifier, RestoreRangeStatus successCriteria, RestoreRange range) {
        return RestoreJobConsistencyChecker.concludeOneRange(RestoreJobConsistencyChecker.populateReplicas(topology), verifier, successCriteria, (Range<Token>)range.tokenRange().range, range.statusByReplica());
    }

    @VisibleForTesting
    static InstanceSetByDc replicaSetForRangeUnsafe(RestoreRange range, TokenRangeReplicasResponse topology) {
        return RestoreJobConsistencyChecker.replicaSetForRange((Range<Token>)range.tokenRange().range, RestoreJobConsistencyChecker.populateReplicas(topology));
    }

    private static /* synthetic */ void lambda$populateStatusByReplica$5(RestoreRange restoreRange, RangeMap updated, Range key, Pair value) {
        HashMap<String, RestoreRangeStatus> statusMap = new HashMap<String, RestoreRangeStatus>((Map)value.getKey());
        statusMap.putAll(restoreRange.statusByReplica());
        updated.put(key, (Object)Pair.of(statusMap, (Object)restoreRange));
    }
}

