/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.operators;

import org.apache.flink.api.common.ExecutionConfig;
import org.apache.flink.api.common.functions.ReduceFunction;
import org.apache.flink.api.common.typeutils.TypeSerializer;
import org.apache.flink.api.common.typeutils.TypeSerializerFactory;
import org.apache.flink.metrics.Counter;
import org.apache.flink.runtime.operators.util.TaskConfig;
import org.apache.flink.runtime.operators.util.metrics.CountingCollector;
import org.apache.flink.util.Collector;
import org.apache.flink.util.MutableObjectIterator;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Reduce task which is executed by a Task Manager. The task has a single input and one or multiple
 * outputs. It is provided with a ReduceFunction implementation.
 *
 * <p>The AllReduceDriver creates an iterator over all records from its input. The elements are
 * handed pairwise to the <code>reduce()</code> method of the ReduceFunction.
 *
 * @see org.apache.flink.api.common.functions.ReduceFunction
 */
public class AllReduceDriver<T> implements Driver<ReduceFunction<T>, T> {

    private static final Logger LOG = LoggerFactory.getLogger(AllReduceDriver.class);

    private TaskContext<ReduceFunction<T>, T> taskContext;

    private MutableObjectIterator<T> input;

    private TypeSerializer<T> serializer;

    private boolean running;

    private boolean objectReuseEnabled = false;

    // ------------------------------------------------------------------------

    @Override
    public void setup(TaskContext<ReduceFunction<T>, T> context) {
        this.taskContext = context;
        this.running = true;
    }

    @Override
    public int getNumberOfInputs() {
        return 1;
    }

    @Override
    public Class<ReduceFunction<T>> getStubType() {
        @SuppressWarnings("unchecked")
        final Class<ReduceFunction<T>> clazz =
                (Class<ReduceFunction<T>>) (Class<?>) ReduceFunction.class;
        return clazz;
    }

    @Override
    public int getNumberOfDriverComparators() {
        return 0;
    }

    // --------------------------------------------------------------------------------------------

    @Override
    public void prepare() throws Exception {
        final TaskConfig config = this.taskContext.getTaskConfig();
        if (config.getDriverStrategy() != DriverStrategy.ALL_REDUCE) {
            throw new Exception(
                    "Unrecognized driver strategy for AllReduce driver: "
                            + config.getDriverStrategy().name());
        }

        TypeSerializerFactory<T> serializerFactory = this.taskContext.getInputSerializer(0);
        this.serializer = serializerFactory.getSerializer();
        this.input = this.taskContext.getInput(0);

        ExecutionConfig executionConfig = taskContext.getExecutionConfig();
        this.objectReuseEnabled = executionConfig.isObjectReuseEnabled();

        if (LOG.isDebugEnabled()) {
            LOG.debug(
                    "AllReduceDriver object reuse: "
                            + (this.objectReuseEnabled ? "ENABLED" : "DISABLED")
                            + ".");
        }
    }

    @Override
    public void run() throws Exception {
        if (LOG.isDebugEnabled()) {
            LOG.debug(
                    this.taskContext.formatLogString(
                            "AllReduce preprocessing done. Running Reducer code."));
        }

        final Counter numRecordsIn =
                this.taskContext.getMetricGroup().getIOMetricGroup().getNumRecordsInCounter();
        final Counter numRecordsOut =
                this.taskContext.getMetricGroup().getIOMetricGroup().getNumRecordsOutCounter();

        final ReduceFunction<T> stub = this.taskContext.getStub();
        final MutableObjectIterator<T> input = this.input;
        final TypeSerializer<T> serializer = this.serializer;
        final Collector<T> collector =
                new CountingCollector<>(this.taskContext.getOutputCollector(), numRecordsOut);

        T val1;
        if ((val1 = input.next()) == null) {
            return;
        }
        numRecordsIn.inc();

        if (objectReuseEnabled) {
            // We only need two objects. The first reference stores results and is
            // eventually collected. New values are read into the second.
            T val2 = serializer.createInstance();

            T value = val1;

            while (running && (val2 = input.next(val2)) != null) {
                numRecordsIn.inc();
                value = stub.reduce(value, val2);

                // we must never read into the object returned
                // by the user, so swap the reuse objects,
                if (value == val2) {
                    T tmp = val1;
                    val1 = val2;
                    val2 = tmp;
                }
            }

            collector.collect(value);
        } else {
            T val2;
            while (running && (val2 = input.next()) != null) {
                numRecordsIn.inc();
                val1 = stub.reduce(val1, val2);
            }

            collector.collect(val1);
        }
    }

    @Override
    public void cleanup() {}

    @Override
    public void cancel() {
        this.running = false;
    }
}
