/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ignite.internal.sql.engine.util.format;

import java.time.Clock;
import java.time.DateTimeException;
import java.time.Year;
import java.util.List;
import java.util.Objects;
import java.util.function.BooleanSupplier;
import java.util.stream.Collectors;
import org.apache.ignite.internal.lang.IgniteStringFormatter;
import org.apache.ignite.internal.sql.engine.util.format.DateTimeFormatElement;
import org.apache.ignite.internal.sql.engine.util.format.DateTimeTemplateField;
import org.apache.ignite.internal.sql.engine.util.format.ParsedFields;
import org.jetbrains.annotations.Nullable;

final class Parser {
    private static final char EOF = '\u0000';
    private final List<DateTimeFormatElement> elements;
    private final StringBuilder buf = new StringBuilder();
    private final Clock clock;
    private int current;
    private int elementIndex;
    private String text;
    private ParsedFields parsedFields;
    @Nullable
    private Hours12 hh12;
    @Nullable
    private TimeZoneFields tz;

    Parser(List<DateTimeFormatElement> elements) {
        this(elements, Clock.systemDefaultZone());
    }

    Parser(List<DateTimeFormatElement> elements, Clock clock) {
        if (elements.isEmpty()) {
            throw new IllegalArgumentException("Element must not be empty");
        }
        Objects.requireNonNull(clock, "clock");
        this.elements = elements;
        this.clock = clock;
    }

    ParsedFields parse(String text) {
        Objects.requireNonNull(text, "text");
        this.current = 0;
        this.buf.setLength(0);
        this.elementIndex = 0;
        this.text = text;
        this.parsedFields = new ParsedFields();
        this.parseElements(text);
        if (this.hh12 != null) {
            assert (this.hh12.clock != 0);
            this.parsedFields.addHours12(this.hh12.clock == 2, this.hh12.value);
        }
        if (this.tz != null) {
            assert (this.tz.parsed != 0);
            this.parsedFields.addTz(this.tz.hours * this.tz.sign, this.tz.minutes * this.tz.sign);
        }
        return this.parsedFields;
    }

    private void parseElements(String text) {
        block4: for (DateTimeFormatElement element : this.elements) {
            if (this.atEnd()) break;
            switch (element.kind) {
                case DELIMITER: {
                    this.delimiter(element);
                    continue block4;
                }
                case FIELD: {
                    this.field(element);
                    continue block4;
                }
            }
            throw new IllegalStateException("Unexpected element kind: " + String.valueOf((Object)element.kind));
        }
        if (this.elementIndex < this.elements.size()) {
            String elems = this.elements.subList(this.elementIndex, this.elements.size()).stream().map(DateTimeFormatElement::toString).collect(Collectors.joining(", "));
            throw Parser.parseError("No values for elements {}", elems);
        }
        if (this.current < text.length()) {
            DateTimeFormatElement e = this.elements.get(this.elementIndex - 1);
            throw Parser.parseError("Unexpected trailing characters after {}", e);
        }
    }

    private boolean atEnd() {
        return this.current >= this.text.length();
    }

    private void delimiter(DateTimeFormatElement element) {
        assert (element.delimiter != '\u0000');
        char c = this.currentChar();
        if (element.delimiter != c) {
            throw Parser.parseError("Invalid format. Expected literal <{}> but got <{}>", Character.valueOf(element.delimiter), Character.valueOf(c));
        }
        this.advancePosition();
        ++this.elementIndex;
    }

    private void field(DateTimeFormatElement element) {
        boolean matches;
        DateTimeTemplateField field = element.template;
        assert (field != null);
        switch (field) {
            case AM: 
            case PM: {
                matches = this.matchHh12(() -> this.matchChars("A.M.") || this.matchChars("P.M."));
                break;
            }
            case HH: 
            case HH12: {
                matches = this.matchHh12(() -> this.matchAtMostDigits(field.maxDigits()));
                break;
            }
            case TZH: {
                matches = this.matchTzField(this::tzh);
                break;
            }
            case TZM: {
                matches = this.matchTzField(() -> this.matchAtMostDigits(field.maxDigits()));
                break;
            }
            default: {
                matches = this.matchAtMostDigits(field.maxDigits());
            }
        }
        if (matches) {
            this.parseFieldValue(field);
            ++this.elementIndex;
        } else {
            throw Parser.parseError("Expected field {} but got <{}>", new Object[]{field, Character.valueOf(this.currentChar())});
        }
    }

    private boolean matchTzField(BooleanSupplier matcher) {
        boolean matches;
        if (this.tz == null) {
            this.tz = new TimeZoneFields();
        }
        if (!(matches = matcher.getAsBoolean())) {
            this.tz = null;
            return false;
        }
        return true;
    }

    private boolean matchHh12(BooleanSupplier matcher) {
        boolean matches;
        if (this.hh12 == null) {
            this.hh12 = new Hours12();
        }
        if (!(matches = matcher.getAsBoolean())) {
            this.hh12 = null;
            return false;
        }
        return true;
    }

    private boolean tzh() {
        int start = this.current;
        if (this.tz == null) {
            this.tz = new TimeZoneFields();
        }
        if (!this.matchSign()) {
            return false;
        }
        this.tz.sign = this.buf.charAt(0) == '+' ? 1 : -1;
        this.buf.setLength(0);
        if (!this.matchAtMostDigits(DateTimeTemplateField.TZH.maxDigits())) {
            this.current = start;
            return false;
        }
        return true;
    }

    private boolean matchSign() {
        char c = this.currentChar();
        if (c == '+' || c == '-') {
            this.addChar();
            this.advancePosition();
            return true;
        }
        return false;
    }

    private boolean matchChars(String chars) {
        int start = this.current;
        for (int i = 0; i < chars.length(); ++i) {
            char c = this.currentChar();
            char p = chars.charAt(i);
            if (Character.toUpperCase(c) != Character.toUpperCase(p)) {
                this.current = start;
                this.buf.setLength(0);
                return false;
            }
            this.addChar();
            this.advancePosition();
        }
        return true;
    }

    private boolean matchAtMostDigits(int n) {
        char c;
        int numDigits;
        for (numDigits = 0; numDigits < n && Character.isDigit(c = this.currentChar()); ++numDigits) {
            this.addChar();
            this.advancePosition();
        }
        return numDigits > 0;
    }

    private char currentChar() {
        if (this.atEnd()) {
            return '\u0000';
        }
        return this.text.charAt(this.current);
    }

    private void advancePosition() {
        ++this.current;
    }

    private void addChar() {
        char c = this.currentChar();
        assert (c != '\u0000') : "Should never read EOF";
        this.buf.append(c);
    }

    private void parseFieldValue(DateTimeTemplateField field) {
        assert (this.buf.length() > 0) : "Field value is empty";
        String value = this.buf.toString();
        this.buf.setLength(0);
        switch (field) {
            case YYYY: {
                this.parseYear(field, value, 0);
                break;
            }
            case YYY: {
                int baseYyy = Year.now(this.clock).getValue() / 1000 * 1000;
                this.parseYear(field, value, baseYyy);
                break;
            }
            case YY: {
                int baseYy = Year.now(this.clock).getValue() / 100 * 100;
                this.parseYear(field, value, baseYy);
                break;
            }
            case Y: {
                int baseY = Year.now(this.clock).getValue() / 10 * 10;
                this.parseYear(field, value, baseY);
                break;
            }
            case MM: {
                this.parseNumber(field, value);
                break;
            }
            case DD: {
                this.parseNumber(field, value);
                break;
            }
            case DDD: {
                this.parseNumber(field, value);
                break;
            }
            case HH: 
            case HH12: {
                this.parse12Hour(field, value);
                break;
            }
            case HH24: {
                this.parseNumber(field, value);
                break;
            }
            case MI: {
                this.parseNumber(field, value);
                break;
            }
            case SS: {
                this.parseNumber(field, value);
                break;
            }
            case SSSSS: {
                this.parseNumber(field, value);
                break;
            }
            case RRRR: 
            case RR: {
                this.parseRoundedYear(field, value);
                break;
            }
            case FF1: 
            case FF3: 
            case FF2: {
                this.parseFaction(field, value, 3, 1000000);
                break;
            }
            case FF4: 
            case FF5: 
            case FF6: {
                this.parseFaction(field, value, 6, 1000);
                break;
            }
            case FF7: 
            case FF8: 
            case FF9: {
                this.parseFaction(field, value, 9, 1);
                break;
            }
            case AM: 
            case PM: {
                this.parseAmPm(value);
                break;
            }
            case TZH: {
                this.parseTimeZone(field, 1, value);
                break;
            }
            case TZM: {
                this.parseTimeZone(field, 2, value);
                break;
            }
            default: {
                throw new IllegalStateException("Unexpected field: " + String.valueOf((Object)field));
            }
        }
    }

    private void parseYear(DateTimeTemplateField field, String value, int base) {
        int v = Parser.parseInt(field, value);
        this.parsedFields.addYear(base, v);
    }

    private void parseRoundedYear(DateTimeTemplateField field, String value) {
        int v = Parser.parseInt(field, value);
        this.parsedFields.addRoundedYear(v);
    }

    private void parseFaction(DateTimeTemplateField field, String value, int len, int multiplier) {
        if (((String)value).length() < len) {
            value = (String)value + "0".repeat(len - ((String)value).length());
        }
        int v = Parser.parseInt(field, (String)value) * multiplier;
        this.parsedFields.add(field.kind(), v);
    }

    private void parseNumber(DateTimeTemplateField field, String value) {
        int v = Parser.parseInt(field, value);
        this.parsedFields.add(field.kind(), v);
    }

    private void parse12Hour(DateTimeTemplateField field, String value) {
        int v = Parser.parseInt(field, value);
        assert (this.hh12 != null) : "hh12 should have been initialized";
        this.hh12.value = v;
    }

    private void parseAmPm(String value) {
        assert (this.hh12 != null) : "hh12 should have been initialized";
        this.hh12.setFlag("P.M.".equalsIgnoreCase(value));
    }

    private void parseTimeZone(DateTimeTemplateField field, int f, String value) {
        int v = Parser.parseInt(field, value);
        assert (this.tz != null) : "tz should have been initialized";
        this.tz.setField(f, v);
    }

    private static int parseInt(DateTimeTemplateField field, String text) {
        int num;
        try {
            num = Integer.parseInt(text);
        }
        catch (NumberFormatException ignore) {
            throw Parser.parseError("Invalid value for {}", field.displayName());
        }
        return num;
    }

    private static DateTimeException parseError(String message, Object ... elements) {
        return new DateTimeException(IgniteStringFormatter.format((String)message, (Object[])elements));
    }

    private static class Hours12 {
        private static final int AM = 1;
        private static final int PM = 2;
        private int clock;
        private int value = -1;

        private Hours12() {
        }

        void setFlag(boolean pm) {
            this.clock = pm ? 2 : 1;
        }
    }

    private static class TimeZoneFields {
        private static final int HOURS = 1;
        private static final int MINUTES = 2;
        private int hours;
        private int minutes;
        private int parsed;
        private int sign;

        private TimeZoneFields() {
        }

        void setField(int field, int value) {
            if (field == 1) {
                assert (this.sign != 0) : "No sign character";
                this.hours = value;
            } else if (field == 2) {
                this.minutes = value;
            } else {
                throw Parser.parseError("Unexpected time zone field", new Object[0]);
            }
            this.parsed |= field;
        }
    }
}

