/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.client.transport;

import java.io.IOException;
import java.nio.channels.AsynchronousCloseException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Queue;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.TimeoutException;
import org.eclipse.jetty.client.Connection;
import org.eclipse.jetty.client.ConnectionPool;
import org.eclipse.jetty.client.Destination;
import org.eclipse.jetty.client.HttpClient;
import org.eclipse.jetty.client.Origin;
import org.eclipse.jetty.client.ProxyConfiguration;
import org.eclipse.jetty.client.Request;
import org.eclipse.jetty.client.Response;
import org.eclipse.jetty.client.RetryableRequestException;
import org.eclipse.jetty.client.transport.HttpExchange;
import org.eclipse.jetty.client.transport.HttpRequest;
import org.eclipse.jetty.client.transport.IConnection;
import org.eclipse.jetty.client.transport.SendFailure;
import org.eclipse.jetty.http.HttpField;
import org.eclipse.jetty.http.HttpHeader;
import org.eclipse.jetty.io.ClientConnectionFactory;
import org.eclipse.jetty.io.CyclicTimeouts;
import org.eclipse.jetty.util.BlockingArrayQueue;
import org.eclipse.jetty.util.Callback;
import org.eclipse.jetty.util.ExceptionUtil;
import org.eclipse.jetty.util.HostPort;
import org.eclipse.jetty.util.NanoTime;
import org.eclipse.jetty.util.Promise;
import org.eclipse.jetty.util.URIUtil;
import org.eclipse.jetty.util.annotation.ManagedAttribute;
import org.eclipse.jetty.util.annotation.ManagedObject;
import org.eclipse.jetty.util.component.ContainerLifeCycle;
import org.eclipse.jetty.util.component.Dumpable;
import org.eclipse.jetty.util.component.DumpableCollection;
import org.eclipse.jetty.util.component.LifeCycle;
import org.eclipse.jetty.util.ssl.SslContextFactory;
import org.eclipse.jetty.util.thread.AutoLock;
import org.eclipse.jetty.util.thread.Scheduler;
import org.eclipse.jetty.util.thread.Sweeper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ManagedObject
public class HttpDestination
extends ContainerLifeCycle
implements Destination,
Callback,
Dumpable,
Sweeper.Sweepable {
    private static final Logger LOG = LoggerFactory.getLogger(HttpDestination.class);
    private final HttpClient client;
    private final Origin origin;
    private final Queue<HttpExchange> exchanges;
    private final ProxyConfiguration.Proxy proxy;
    private final ClientConnectionFactory connectionFactory;
    private final HttpField hostField;
    private final RequestTimeouts requestTimeouts;
    private final AutoLock staleLock = new AutoLock();
    private ConnectionPool connectionPool;
    private boolean stale;
    private long activeNanoTime;

    public HttpDestination(HttpClient client, Origin origin) {
        this.client = client;
        this.origin = origin;
        this.exchanges = this.newExchangeQueue(client);
        this.requestTimeouts = new RequestTimeouts(client.getScheduler());
        Object host = HostPort.normalizeHost(this.getHost());
        int port = this.getPort();
        String scheme = this.getScheme();
        if (port != URIUtil.getDefaultPortForScheme(scheme)) {
            host = (String)host + ":" + port;
        }
        this.hostField = new HttpField(HttpHeader.HOST, (String)host);
        ProxyConfiguration proxyConfig = client.getProxyConfiguration();
        this.proxy = proxyConfig.match(origin);
        ClientConnectionFactory connectionFactory = client.getHttpClientTransport();
        if (this.proxy != null) {
            connectionFactory = this.proxy.newClientConnectionFactory(connectionFactory);
        }
        this.connectionFactory = connectionFactory;
    }

    public void accept(Connection connection) {
        this.connectionPool.accept(connection);
    }

    public boolean stale() {
        try (AutoLock ignored = this.staleLock.lock();){
            boolean stale = this.stale;
            if (!stale) {
                this.activeNanoTime = NanoTime.now();
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("Stale check done with result {} on {}", (Object)stale, (Object)this);
            }
            boolean bl = stale;
            return bl;
        }
    }

    @Override
    public boolean sweep() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Sweep check in progress on {}", (Object)this);
        }
        boolean remove = false;
        try (AutoLock ignored = this.staleLock.lock();){
            boolean stale;
            boolean bl = stale = this.exchanges.isEmpty() && this.connectionPool.isEmpty();
            if (!stale) {
                this.activeNanoTime = NanoTime.now();
            } else if (NanoTime.millisSince(this.activeNanoTime) >= this.getHttpClient().getDestinationIdleTimeout()) {
                this.stale = true;
                remove = true;
            }
        }
        if (remove) {
            this.getHttpClient().removeDestination(this);
            LifeCycle.stop(this);
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Sweep check done with result {} on {}", (Object)remove, (Object)this);
        }
        return remove;
    }

    @Override
    protected void doStart() throws Exception {
        this.connectionPool = this.newConnectionPool(this.client);
        this.addBean(this.connectionPool, true);
        super.doStart();
        Sweeper connectionPoolSweeper = this.client.getBean(Sweeper.class);
        if (connectionPoolSweeper != null && this.connectionPool instanceof Sweeper.Sweepable) {
            connectionPoolSweeper.offer((Sweeper.Sweepable)((Object)this.connectionPool));
        }
    }

    @Override
    protected void doStop() throws Exception {
        this.requestTimeouts.destroy();
        this.abortExchanges(new AsynchronousCloseException());
        Sweeper connectionPoolSweeper = this.client.getBean(Sweeper.class);
        if (connectionPoolSweeper != null && this.connectionPool instanceof Sweeper.Sweepable) {
            connectionPoolSweeper.remove((Sweeper.Sweepable)((Object)this.connectionPool));
        }
        super.doStop();
        this.removeBean(this.connectionPool);
    }

    protected ConnectionPool newConnectionPool(HttpClient client) {
        return client.getHttpClientTransport().getConnectionPoolFactory().newConnectionPool(this);
    }

    protected Queue<HttpExchange> newExchangeQueue(HttpClient client) {
        int maxCapacity = client.getMaxRequestsQueuedPerDestination();
        if (maxCapacity > 32) {
            return BlockingArrayQueue.newInstance(32, maxCapacity);
        }
        return new BlockingArrayQueue<HttpExchange>(maxCapacity);
    }

    @Override
    @ManagedAttribute(value="Whether this destination is secure")
    public boolean isSecure() {
        return this.origin.isSecure();
    }

    @Override
    @ManagedAttribute(value="The HttpClient")
    public HttpClient getHttpClient() {
        return this.client;
    }

    @Override
    public Origin getOrigin() {
        return this.origin;
    }

    public Queue<HttpExchange> getHttpExchanges() {
        return this.exchanges;
    }

    @Override
    public ProxyConfiguration.Proxy getProxy() {
        return this.proxy;
    }

    public Origin resolveOrigin() {
        return this.proxy == null ? this.getOrigin() : this.proxy.getOrigin();
    }

    public ClientConnectionFactory resolveClientConnectionFactory() {
        return this.connectionFactory;
    }

    public SslContextFactory.Client resolveSslContextFactory() {
        boolean secure;
        boolean bl = secure = this.proxy == null ? this.isSecure() : this.proxy.isSecure();
        if (secure) {
            SslContextFactory.Client sslContextFactory = this.getHttpClient().getSslContextFactory();
            if (this.proxy == null) {
                return sslContextFactory;
            }
            SslContextFactory.Client proxySslContextFactory = this.proxy.getSslContextFactory();
            return proxySslContextFactory != null ? proxySslContextFactory : sslContextFactory;
        }
        return null;
    }

    @ManagedAttribute(value="The destination scheme")
    public String getScheme() {
        return this.getOrigin().getScheme();
    }

    @ManagedAttribute(value="The destination host")
    public String getHost() {
        return this.getOrigin().getAddress().getHost();
    }

    @ManagedAttribute(value="The destination port")
    public int getPort() {
        return this.getOrigin().getAddress().getPort();
    }

    @ManagedAttribute(value="The number of queued requests")
    public int getQueuedRequestCount() {
        return this.exchanges.size();
    }

    public HttpField getHostField() {
        return this.hostField;
    }

    @Override
    @ManagedAttribute(value="The connection pool")
    public ConnectionPool getConnectionPool() {
        return this.connectionPool;
    }

    @Override
    public void succeeded() {
        this.send(false);
    }

    @Override
    public void failed(Throwable x) {
        this.abortExchanges(x);
    }

    @Override
    public void send(Request request, Response.CompleteListener listener) {
        ((HttpRequest)request).sendAsync(this, listener);
    }

    void send(HttpRequest request) {
        this.send(new HttpExchange(this, request));
    }

    public void send(HttpExchange exchange) {
        HttpRequest request = exchange.getRequest();
        if (this.client.isRunning()) {
            if (this.enqueue(this.exchanges, exchange)) {
                request.sent();
                this.requestTimeouts.schedule(exchange);
                if (!this.client.isRunning() && this.exchanges.remove(exchange)) {
                    request.abort(new RejectedExecutionException(String.valueOf(this.client) + " is stopping"));
                } else {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Queued {} for {}", (Object)request, (Object)this);
                    }
                    if (!request.getAndSetQueued()) {
                        request.notifyQueued();
                    }
                    this.send();
                }
            } else {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Max queue size {} exceeded by {} for {}", this.client.getMaxRequestsQueuedPerDestination(), request, this);
                }
                request.abort(new RejectedExecutionException("Max requests queued per destination " + this.client.getMaxRequestsQueuedPerDestination() + " exceeded for " + String.valueOf(this)));
            }
        } else {
            request.abort(new RejectedExecutionException(String.valueOf(this.client) + " is stopped"));
        }
    }

    protected boolean enqueue(Queue<HttpExchange> queue, HttpExchange exchange) {
        return queue.offer(exchange);
    }

    public void send() {
        this.send(true);
    }

    private void send(boolean create) {
        if (!this.getHttpExchanges().isEmpty()) {
            this.process(create);
        }
    }

    private void process(boolean create) {
        boolean proceed;
        Connection connection;
        while ((connection = this.connectionPool.acquire(create)) != null && (proceed = this.process(connection))) {
            create = false;
        }
    }

    private boolean process(Connection connection) {
        HttpClient client = this.getHttpClient();
        HttpExchange exchange = this.getHttpExchanges().poll();
        if (LOG.isDebugEnabled()) {
            LOG.debug("Processing exchange {} on {} of {}", exchange, connection, this);
        }
        if (exchange == null) {
            this.releaseOrClose(connection);
            if (!client.isRunning()) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("{} is stopping", (Object)client);
                }
                connection.close();
            }
            return false;
        }
        HttpRequest request = exchange.getRequest();
        Throwable cause = request.getAbortCause();
        if (cause != null) {
            if (LOG.isDebugEnabled()) {
                LOG.atDebug().setCause(cause).log("Aborted before processing {}", (Object)exchange);
            }
            exchange.abort(cause, Promise.from(() -> this.releaseOrClose(connection)));
            return this.getQueuedRequestCount() > 0;
        }
        SendFailure failure = this.send((IConnection)connection, exchange);
        if (failure == null) {
            return this.getQueuedRequestCount() > 0;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Send failed {} for {}", (Object)failure, (Object)exchange);
        }
        if (failure.retry) {
            try {
                HttpDestination newDestination = (HttpDestination)client.resolveDestination(request);
                newDestination.send(exchange);
                this.releaseOrClose(connection);
                return false;
            }
            catch (Throwable x) {
                ExceptionUtil.addSuppressedIfNotAssociated(x, failure.failure);
                failure = new SendFailure(x, false);
            }
        }
        request.abort(failure.failure);
        this.releaseOrClose(connection);
        return this.getQueuedRequestCount() > 0;
    }

    protected SendFailure send(IConnection connection, HttpExchange exchange) {
        try {
            return connection.send(exchange);
        }
        catch (Throwable x) {
            return new SendFailure(x, x instanceof RetryableRequestException);
        }
    }

    @Override
    public void newConnection(Promise<Connection> promise) {
        this.createConnection(promise);
    }

    protected void createConnection(Promise<Connection> promise) {
        this.client.newConnection(this, promise);
    }

    public boolean remove(HttpExchange exchange) {
        return this.exchanges.remove(exchange);
    }

    public void release(Connection connection) {
        HttpClient client = this.getHttpClient();
        if (client.isRunning()) {
            if (this.connectionPool.isActive(connection)) {
                boolean released = this.releaseOrClose(connection);
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Released: {} {}", (Object)released, (Object)connection);
                }
                this.send(!released);
            } else if (LOG.isDebugEnabled()) {
                LOG.debug("Released explicit {}", (Object)connection);
            }
        } else {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Closing {}, {} is stopped", (Object)connection, (Object)client);
            }
            connection.close();
        }
    }

    private boolean releaseOrClose(Connection connection) {
        boolean released = this.connectionPool.release(connection);
        if (!released) {
            connection.close();
        }
        return released;
    }

    public boolean remove(Connection connection) {
        boolean removed = this.connectionPool.remove(connection);
        if (removed) {
            this.send(true);
        }
        return removed;
    }

    private void abortExchanges(Throwable cause) {
        for (HttpExchange exchange : new ArrayList<HttpExchange>(this.exchanges)) {
            exchange.getRequest().abort(cause);
        }
    }

    @Override
    public void dump(Appendable out, String indent) throws IOException {
        this.dumpObjects(out, indent, new DumpableCollection("exchanges", this.exchanges));
    }

    public String asString() {
        return this.getOrigin().asString();
    }

    @ManagedAttribute(value="For how long this destination has been idle in ms")
    public long getIdle() {
        if (this.getHttpClient().getDestinationIdleTimeout() <= 0L) {
            return -1L;
        }
        try (AutoLock ignored = this.staleLock.lock();){
            long l = NanoTime.millisSince(this.activeNanoTime);
            return l;
        }
    }

    @ManagedAttribute(value="Whether this destinations is stale")
    public boolean isStale() {
        try (AutoLock ignored = this.staleLock.lock();){
            boolean bl = this.stale;
            return bl;
        }
    }

    @Override
    public String toString() {
        return String.format("%s[%s]@%x%s,state=%s,queue=%d,pool=%s,stale=%b,idle=%d", HttpDestination.class.getSimpleName(), this.getOrigin(), this.hashCode(), this.proxy == null ? "" : "(via " + String.valueOf(this.proxy) + ")", this.getState(), this.getQueuedRequestCount(), this.getConnectionPool(), this.isStale(), this.getIdle());
    }

    private class RequestTimeouts
    extends CyclicTimeouts<HttpExchange> {
        private RequestTimeouts(Scheduler scheduler) {
            super(scheduler);
        }

        @Override
        protected Iterator<HttpExchange> iterator() {
            return HttpDestination.this.exchanges.iterator();
        }

        @Override
        protected boolean onExpired(HttpExchange exchange) {
            HttpRequest request = exchange.getRequest();
            request.abort(new TimeoutException("Total timeout " + request.getConversation().getTimeout() + " ms elapsed"));
            return false;
        }
    }
}

