/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs;

import java.io.Closeable;
import java.io.IOException;
import java.security.PrivilegedExceptionAction;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.function.Supplier;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.SafeModeAction;
import org.apache.hadoop.fs.permission.FsAction;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hdfs.DFSTestUtil;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.protocol.ErasureCodingPolicy;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockManagerTestUtil;
import org.apache.hadoop.hdfs.server.common.HdfsServerConstants;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.NameNode;
import org.apache.hadoop.hdfs.server.namenode.NameNodeAdapter;
import org.apache.hadoop.hdfs.server.namenode.SafeModeException;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.io.erasurecode.ECSchema;
import org.apache.hadoop.ipc.RemoteException;
import org.apache.hadoop.metrics2.MetricsRecordBuilder;
import org.apache.hadoop.security.AccessControlException;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.test.GenericTestUtils;
import org.apache.hadoop.test.MetricsAsserts;
import org.apache.hadoop.thirdparty.com.google.common.collect.Lists;
import org.apache.hadoop.util.StringUtils;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TestSafeMode {
    public static final Logger LOG = LoggerFactory.getLogger(TestSafeMode.class);
    private static final Path TEST_PATH = new Path("/test");
    private static final int BLOCK_SIZE = 1024;
    private static final String NEWLINE = System.getProperty("line.separator");
    Configuration conf;
    MiniDFSCluster cluster;
    FileSystem fs;
    DistributedFileSystem dfs;
    private static final String NN_METRICS = "NameNodeActivity";

    @Before
    public void startUp() throws IOException {
        this.conf = new HdfsConfiguration();
        this.conf.setInt("dfs.blocksize", 1024);
        this.conf.setBoolean("dfs.namenode.acls.enabled", true);
        this.conf.setBoolean("dfs.namenode.xattrs.enabled", true);
        this.cluster = new MiniDFSCluster.Builder(this.conf).numDataNodes(1).build();
        this.cluster.waitActive();
        this.fs = this.cluster.getFileSystem();
        this.dfs = (DistributedFileSystem)this.fs;
    }

    @After
    public void tearDown() throws IOException {
        if (this.fs != null) {
            this.fs.close();
            this.fs = null;
        }
        if (this.cluster != null) {
            this.cluster.shutdown();
            this.cluster = null;
        }
    }

    @Test
    public void testManualSafeMode() throws IOException {
        this.fs = this.cluster.getFileSystem();
        Path file1 = new Path("/tmp/testManualSafeMode/file1");
        Path file2 = new Path("/tmp/testManualSafeMode/file2");
        DFSTestUtil.createFile(this.fs, file1, 1000L, (short)1, 0L);
        DFSTestUtil.createFile(this.fs, file2, 1000L, (short)1, 0L);
        this.fs.close();
        this.cluster.shutdown();
        this.cluster = new MiniDFSCluster.Builder(this.conf).numDataNodes(0).format(false).build();
        this.cluster.waitActive();
        this.dfs = this.cluster.getFileSystem();
        Assert.assertTrue((String)"No datanode is started. Should be in SafeMode", (boolean)this.dfs.setSafeMode(SafeModeAction.GET));
        this.dfs.setSafeMode(SafeModeAction.ENTER);
        this.cluster.startDataNodes(this.conf, 1, true, null, null);
        this.cluster.waitActive();
        try {
            Thread.sleep(2000L);
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
        Assert.assertTrue((String)"should still be in SafeMode", (boolean)this.dfs.setSafeMode(SafeModeAction.GET));
        Assert.assertFalse((String)"should not be in SafeMode", (boolean)this.dfs.setSafeMode(SafeModeAction.LEAVE));
    }

    @Test(timeout=45000L)
    public void testNoExtensionIfNoBlocks() throws IOException {
        this.cluster.getConfiguration(0).setInt("dfs.namenode.safemode.extension", 60000);
        this.cluster.restartNameNode(new String[0]);
        String status = this.cluster.getNameNode().getNamesystem().getSafemode();
        Assert.assertEquals((Object)"", (Object)status);
    }

    @Test(timeout=45000L)
    public void testInitializeReplQueuesEarly() throws Exception {
        LOG.info("Starting testInitializeReplQueuesEarly");
        BlockManagerTestUtil.setWritingPrefersLocalNode(this.cluster.getNamesystem().getBlockManager(), false);
        this.cluster.startDataNodes(this.conf, 2, true, HdfsServerConstants.StartupOption.REGULAR, null);
        this.cluster.waitActive();
        LOG.info("Creating files");
        DFSTestUtil.createFile(this.fs, TEST_PATH, 15360L, (short)1, 1L);
        LOG.info("Stopping all DataNodes");
        LinkedList dnprops = Lists.newLinkedList();
        dnprops.add(this.cluster.stopDataNode(0));
        dnprops.add(this.cluster.stopDataNode(0));
        dnprops.add(this.cluster.stopDataNode(0));
        this.cluster.getConfiguration(0).setFloat("dfs.namenode.replqueue.threshold-pct", 0.06666667f);
        LOG.info("Restarting NameNode");
        this.cluster.restartNameNode(new String[0]);
        NameNode nn = this.cluster.getNameNode();
        String status = nn.getNamesystem().getSafemode();
        Assert.assertEquals((Object)("Safe mode is ON. The reported blocks 0 needs additional 14 blocks to reach the threshold 0.9990 of total blocks 15." + NEWLINE + "The minimum number of live datanodes is not required. Safe mode will be turned off automatically once the thresholds have been reached."), (Object)status);
        Assert.assertFalse((String)"Mis-replicated block queues should not be initialized until threshold is crossed", (boolean)NameNodeAdapter.safeModeInitializedReplQueues(nn));
        LOG.info("Restarting one DataNode");
        this.cluster.restartDataNode((MiniDFSCluster.DataNodeProperties)dnprops.remove(0));
        GenericTestUtils.waitFor((Supplier)new Supplier<Boolean>(){

            @Override
            public Boolean get() {
                return MetricsAsserts.getLongCounter((String)"StorageBlockReportNumOps", (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)TestSafeMode.NN_METRICS)) == (long)TestSafeMode.this.cluster.getStoragesPerDatanode();
            }
        }, (long)10L, (long)10000L);
        long safe = NameNodeAdapter.getSafeModeSafeBlocks(nn);
        Assert.assertTrue((String)"Expected first block report to make some blocks safe.", (safe > 0L ? 1 : 0) != 0);
        Assert.assertTrue((String)"Did not expect first block report to make all blocks safe.", (safe < 15L ? 1 : 0) != 0);
        Assert.assertTrue((boolean)NameNodeAdapter.safeModeInitializedReplQueues(nn));
        BlockManagerTestUtil.updateState(nn.getNamesystem().getBlockManager());
        long underReplicatedBlocks = nn.getNamesystem().getUnderReplicatedBlocks();
        while (underReplicatedBlocks != 15L - safe) {
            LOG.info("UnderReplicatedBlocks expected=" + (15L - safe) + ", actual=" + underReplicatedBlocks);
            Thread.sleep(100L);
            BlockManagerTestUtil.updateState(nn.getNamesystem().getBlockManager());
            underReplicatedBlocks = nn.getNamesystem().getUnderReplicatedBlocks();
        }
        this.cluster.restartDataNodes();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testRbwBlocksNotConsideredUnderReplicated() throws IOException {
        ArrayList stms = Lists.newArrayList();
        try {
            DFSTestUtil.createFile(this.fs, new Path("/junk-blocks"), 4096L, (short)1, 1L);
            for (int i = 0; i < 10; ++i) {
                FSDataOutputStream stm = this.fs.create(new Path("/append-" + i), true, 1024, (short)1, 1024L);
                stms.add(stm);
                stm.write(1);
                stm.hflush();
            }
            this.cluster.restartNameNode(new String[0]);
            FSNamesystem ns = this.cluster.getNameNode(0).getNamesystem();
            BlockManagerTestUtil.updateState(ns.getBlockManager());
            Assert.assertEquals((long)0L, (long)ns.getPendingReplicationBlocks());
            Assert.assertEquals((long)0L, (long)ns.getCorruptReplicaBlocks());
            Assert.assertEquals((long)0L, (long)ns.getMissingBlocksCount());
        }
        finally {
            for (FSDataOutputStream stm : stms) {
                IOUtils.closeStream((Closeable)stm);
            }
            this.cluster.shutdown();
        }
    }

    public void runFsFun(String msg, FSRun f) {
        try {
            f.run(this.fs);
            Assert.fail((String)msg);
        }
        catch (RemoteException re) {
            Assert.assertEquals((Object)SafeModeException.class.getName(), (Object)re.getClassName());
            GenericTestUtils.assertExceptionContains((String)"Name node is in safe mode", (Throwable)re);
        }
        catch (SafeModeException re) {
        }
        catch (IOException ioe) {
            Assert.fail((String)(msg + " " + StringUtils.stringifyException((Throwable)ioe)));
        }
    }

    @Test
    public void testSafeModeExceptionText() throws Exception {
        final Path file1 = new Path("/file1");
        DFSTestUtil.createFile(this.fs, file1, 1024L, (short)1, 0L);
        Assert.assertTrue((String)"Could not enter SM", (boolean)this.dfs.setSafeMode(SafeModeAction.ENTER));
        try {
            FSRun fsRun = new FSRun(){

                @Override
                public void run(FileSystem fileSystem) throws IOException {
                    ((DistributedFileSystem)fileSystem).setQuota(file1, 1L, 1L);
                }
            };
            fsRun.run(this.fs);
            Assert.fail((String)"Should not succeed with no exceptions!");
        }
        catch (RemoteException re) {
            Assert.assertEquals((Object)SafeModeException.class.getName(), (Object)re.getClassName());
            GenericTestUtils.assertExceptionContains((String)NameNode.getServiceAddress((Configuration)this.conf, (boolean)true).getHostName(), (Throwable)re);
        }
        catch (IOException ioe) {
            Assert.fail((String)("Encountered exception " + StringUtils.stringifyException((Throwable)ioe)));
        }
    }

    @Test
    public void testOperationsWhileInSafeMode() throws IOException, InterruptedException {
        final Path file1 = new Path("/file1");
        Assert.assertFalse((boolean)this.dfs.setSafeMode(SafeModeAction.GET));
        DFSTestUtil.createFile(this.fs, file1, 1024L, (short)1, 0L);
        Assert.assertTrue((String)"Could not enter SM", (boolean)this.dfs.setSafeMode(SafeModeAction.ENTER));
        this.runFsFun("Set quota while in SM", new FSRun(){

            @Override
            public void run(FileSystem fs) throws IOException {
                ((DistributedFileSystem)fs).setQuota(file1, 1L, 1L);
            }
        });
        this.runFsFun("Set perm while in SM", new FSRun(){

            @Override
            public void run(FileSystem fs) throws IOException {
                fs.setPermission(file1, FsPermission.getDefault());
            }
        });
        this.runFsFun("Set owner while in SM", new FSRun(){

            @Override
            public void run(FileSystem fs) throws IOException {
                fs.setOwner(file1, "user", "group");
            }
        });
        this.runFsFun("Set repl while in SM", new FSRun(){

            @Override
            public void run(FileSystem fs) throws IOException {
                fs.setReplication(file1, (short)1);
            }
        });
        this.runFsFun("Append file while in SM", new FSRun(){

            @Override
            public void run(FileSystem fs) throws IOException {
                DFSTestUtil.appendFile(fs, file1, "new bytes");
            }
        });
        this.runFsFun("Truncate file while in SM", new FSRun(){

            @Override
            public void run(FileSystem fs) throws IOException {
                fs.truncate(file1, 0L);
            }
        });
        this.runFsFun("Delete file while in SM", new FSRun(){

            @Override
            public void run(FileSystem fs) throws IOException {
                fs.delete(file1, false);
            }
        });
        this.runFsFun("Rename file while in SM", new FSRun(){

            @Override
            public void run(FileSystem fs) throws IOException {
                fs.rename(file1, new Path("file2"));
            }
        });
        this.runFsFun("Set time while in SM", new FSRun(){

            @Override
            public void run(FileSystem fs) throws IOException {
                fs.setTimes(file1, 0L, 0L);
            }
        });
        this.runFsFun("modifyAclEntries while in SM", new FSRun(){

            @Override
            public void run(FileSystem fs) throws IOException {
                fs.modifyAclEntries(file1, (List)Lists.newArrayList());
            }
        });
        this.runFsFun("removeAclEntries while in SM", new FSRun(){

            @Override
            public void run(FileSystem fs) throws IOException {
                fs.removeAclEntries(file1, (List)Lists.newArrayList());
            }
        });
        this.runFsFun("removeDefaultAcl while in SM", new FSRun(){

            @Override
            public void run(FileSystem fs) throws IOException {
                fs.removeDefaultAcl(file1);
            }
        });
        this.runFsFun("removeAcl while in SM", new FSRun(){

            @Override
            public void run(FileSystem fs) throws IOException {
                fs.removeAcl(file1);
            }
        });
        this.runFsFun("setAcl while in SM", new FSRun(){

            @Override
            public void run(FileSystem fs) throws IOException {
                fs.setAcl(file1, (List)Lists.newArrayList());
            }
        });
        this.runFsFun("setXAttr while in SM", new FSRun(){

            @Override
            public void run(FileSystem fs) throws IOException {
                fs.setXAttr(file1, "user.a1", null);
            }
        });
        this.runFsFun("removeXAttr while in SM", new FSRun(){

            @Override
            public void run(FileSystem fs) throws IOException {
                fs.removeXAttr(file1, "user.a1");
            }
        });
        try {
            DFSTestUtil.readFile(this.fs, file1);
        }
        catch (IOException ioe) {
            Assert.fail((String)"Set times failed while in SM");
        }
        try {
            this.fs.getAclStatus(file1);
        }
        catch (IOException ioe) {
            Assert.fail((String)"getAclStatus failed while in SM");
        }
        UserGroupInformation ugiX = UserGroupInformation.createRemoteUser((String)"userX");
        FileSystem myfs = (FileSystem)ugiX.doAs((PrivilegedExceptionAction)new PrivilegedExceptionAction<FileSystem>(){

            @Override
            public FileSystem run() throws IOException {
                return FileSystem.get((Configuration)TestSafeMode.this.conf);
            }
        });
        myfs.access(file1, FsAction.READ);
        try {
            myfs.access(file1, FsAction.WRITE);
            Assert.fail((String)"The access call should have failed.");
        }
        catch (AccessControlException accessControlException) {
            // empty catch block
        }
        ECSchema toAddSchema = new ECSchema("testcodec", 3, 2);
        ErasureCodingPolicy newPolicy = new ErasureCodingPolicy(toAddSchema, 131072);
        ErasureCodingPolicy[] policyArray = new ErasureCodingPolicy[]{newPolicy};
        try {
            this.dfs.addErasureCodingPolicies(policyArray);
            Assert.fail((String)"AddErasureCodingPolicies should have failed.");
        }
        catch (IOException ioe) {
            GenericTestUtils.assertExceptionContains((String)"Cannot add erasure coding policy", (Throwable)ioe);
        }
        try {
            this.dfs.removeErasureCodingPolicy("testECName");
            Assert.fail((String)"RemoveErasureCodingPolicy should have failed.");
        }
        catch (IOException ioe) {
            GenericTestUtils.assertExceptionContains((String)"Cannot remove erasure coding policy", (Throwable)ioe);
        }
        Assert.assertFalse((String)"Could not leave SM", (boolean)this.dfs.setSafeMode(SafeModeAction.LEAVE));
    }

    @Test
    public void testDatanodeThreshold() throws IOException {
        this.cluster.shutdown();
        Configuration conf = this.cluster.getConfiguration(0);
        conf.setInt("dfs.namenode.safemode.extension", 0);
        conf.setInt("dfs.namenode.safemode.min.datanodes", 1);
        this.cluster.restartNameNode(new String[0]);
        this.fs = this.cluster.getFileSystem();
        String tipMsg = this.cluster.getNamesystem().getSafemode();
        Assert.assertTrue((String)("Safemode tip message doesn't look right: " + tipMsg), (boolean)tipMsg.contains("The number of live datanodes 0 needs an additional 1 live datanodes to reach the minimum number 1." + NEWLINE + "Safe mode will be turned off automatically"));
        this.cluster.startDataNodes(conf, 1, true, null, null);
        try {
            Thread.sleep(1000L);
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
        Assert.assertEquals((Object)"", (Object)this.cluster.getNamesystem().getSafemode());
    }

    @Test
    public void testSafeModeUtils() throws IOException {
        this.dfs = this.cluster.getFileSystem();
        this.dfs.setSafeMode(SafeModeAction.ENTER);
        Assert.assertTrue((String)"State was expected to be in safemode.", (boolean)this.dfs.isInSafeMode());
        this.dfs.setSafeMode(SafeModeAction.LEAVE);
        Assert.assertFalse((String)"State was expected to be out of safemode.", (boolean)this.dfs.isInSafeMode());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testSafeModeWhenZeroBlockLocations() throws IOException {
        try {
            NameNode namenode;
            Path file1;
            block8: {
                file1 = new Path("/tmp/testManualSafeMode/file1");
                Path file2 = new Path("/tmp/testManualSafeMode/file2");
                System.out.println("Created file1 and file2.");
                DFSTestUtil.createFile(this.fs, file1, 1000L, (short)1, 0L);
                DFSTestUtil.createFile(this.fs, file2, 2000L, (short)1, 0L);
                this.checkGetBlockLocationsWorks(this.fs, file1);
                namenode = this.cluster.getNameNode();
                this.dfs.setSafeMode(SafeModeAction.ENTER);
                Assert.assertTrue((String)"should still be in SafeMode", (boolean)namenode.isInSafeMode());
                this.checkGetBlockLocationsWorks(this.fs, file1);
                this.dfs.setSafeMode(SafeModeAction.LEAVE);
                Assert.assertFalse((String)"should not be in SafeMode", (boolean)namenode.isInSafeMode());
                this.cluster.shutdownDataNodes();
                this.cluster.shutdownNameNode(0);
                this.cluster.restartNameNode(new String[0]);
                this.cluster.waitActive();
                System.out.println("Restarted cluster with just the NameNode");
                namenode = this.cluster.getNameNode();
                Assert.assertTrue((String)"No datanode is started. Should be in SafeMode", (boolean)namenode.isInSafeMode());
                FileStatus stat = this.fs.getFileStatus(file1);
                try {
                    this.fs.getFileBlockLocations(stat, 0L, 1000L);
                    Assert.assertTrue((String)"Should have got safemode exception", (boolean)false);
                }
                catch (SafeModeException safeModeException) {
                }
                catch (RemoteException re) {
                    if (re.getClassName().equals(SafeModeException.class.getName())) break block8;
                    Assert.assertTrue((String)"Should have got safemode exception", (boolean)false);
                }
            }
            this.dfs.setSafeMode(SafeModeAction.LEAVE);
            Assert.assertFalse((String)"Should not be in safemode", (boolean)namenode.isInSafeMode());
            this.checkGetBlockLocationsWorks(this.fs, file1);
        }
        finally {
            if (this.fs != null) {
                this.fs.close();
            }
            if (this.cluster != null) {
                this.cluster.shutdown();
            }
        }
    }

    void checkGetBlockLocationsWorks(FileSystem fs, Path fileName) throws IOException {
        FileStatus stat = fs.getFileStatus(fileName);
        try {
            fs.getFileBlockLocations(stat, 0L, 1000L);
        }
        catch (SafeModeException e) {
            Assert.assertTrue((String)"Should have not got safemode exception", (boolean)false);
        }
        catch (RemoteException re) {
            Assert.assertTrue((String)"Should have not got remote exception", (boolean)false);
        }
    }

    public static interface FSRun {
        public void run(FileSystem var1) throws IOException;
    }
}

